function refine(iname, oname, tname)
% refine - read rules from a plain text file.  For each rule, generate
% a Maxima progam that will refine that rule - calculating its
% coefficients and weights to a precision of 36 decimal digits.
% USAGE: refine(iname, oname, tname)
% INPUTS
%   iname = name of input file
%   oname = optional name of the Maxima program, default "refineXXX.mac",
%           where XXX is the root name of the input file.
%   tname = optional name of file written by the Maxima program, default "refinedXXX.txt"
% OUTPUTS
%   refineXXX.mac - the Maxima program,
% where XXX is the root of the input file name

ix=regexp(iname,'\..*$');
if ~isempty(ix)
    root=iname(1:ix-1);
end

if nargin<2
        oname = sprintf('refine%s.mac',root);% Maxima program
end

if nargin<3
        tname = sprintf('refined%s.txt',root);% file to collect all the refined rules
end

rules=evaluate(iname,-1);

fprintf('%s creating %s\n',iname,oname);
of=fopen(oname,'w');

for ir=1:length(rules)
    rule=rules(ir);

    x=rule.x;
    n=rule.d;                           % dimensions
    N=rule.n;                           % # points
    deg=rule.o;                         % degree
    pp = rule.alpha;                    % powers for monomials correctly integrated

    pstring=rule.problem;
    problems={'G','E2','E1','S','U'};
    for ip=1:length(problems)
        if strcmp(pstring,problems(ip))
            found=1;
            break;
        end
    end
    if ~found
        error('unrecognized problem string "%s"\n', pstring);
        return
    end

    fprintf('formula %d, rule: "%s %d_%d_%d";\n', ir, rule.problem, n, N, deg);

    fprintf(of, '/* %s:%d formula %d */\n', iname, rule.line, ir);
    fprintf(of, 'rule:"%d_%d_%d";\n', n, N, deg);
    fprintf(of, 'n:%d;\n',n);
    fprintf(of, 'N:%d;\n',N);
    fprintf(of, 'deg:%d;\n',deg);

    fprintf(of, 'fpprintprec:32;\n');
    fprintf(of, 'fpprec:fpprintprec*deg+10;\n');

    fprintf(of, 'remvalue(a,p,');
    for i=1:N*(n+1)
        fprintf(of,'x%d',i);
        if i==N*(n+1), break; end
        fprintf(of, ',');
        if mod(i,10)==9, fprintf(of, '\n'); end
    end
    fprintf(of, ');\n');

    fprintf(of, 'x:[');
    for i=1:N*(n+1)
        fprintf(of,'x%d',i);
        if i==N*(n+1), break; end
        fprintf(of, ',');
        if mod(i,10)==9, fprintf(of, '\n'); end
    end
    fprintf(of, '];\n');

    fprintf(of,'pp:matrix(\n');
    for i=1:size(pp,1)
        fprintf(of, '  [');
        for j=1:n-1
            fprintf(of, '%d, ',pp(i,j));
        end

        if i<size(pp,1), 
            fprintf(of,'%d],\n', pp(i,n));
        else
            fprintf(of, '%d]\n', pp(i,n));
        end
    end
    fprintf(of, ');\n');

    fprintf(of, 'for problem in [%d] do block(\n', ip);
    fprintf(of, 'ex(a,p) := if p=0 then 1 else a^p,\n');
    fprintf(of, 'eq:[],\n');
    %fprintf(of, 'alpha:makelist(0,i,n),\n');
    fprintf(of, 'for kk:1 step 1 thru length(pp) do block(\n');
    fprintf(of, '  alpha:list_matrix_entries(row(pp,kk)),\n');
    fprintf(of, '  beta:(alpha+makelist(1,ii,n))/2,\n');
    fprintf(of, '  odd:false,\n');
    fprintf(of, '  for ii:1 thru n do odd:odd or oddp(alpha[ii]),\n');
    fprintf(of, '  I: if odd then 0                                                                                                         else\n');
    fprintf(of, '  if problem=1 /* G  */ then 2^sum(beta[ii],ii,1,n)/(2*%%pi)^(n/2)*product(gamma(beta[ii]),ii,1,n)                          else\n');
    fprintf(of, '  if problem=2 /* E2 */ then product(gamma(beta[ii]),ii,1,n)                                                               else\n');
    fprintf(of, '  if problem=3 /* E1 */ then 2*gamma(n+sum(alpha[ii],ii,1,n))*product(gamma(beta[ii]),ii,1,n)/gamma(sum(beta[ii],ii,1,n))  else\n');
    fprintf(of, '  if problem=4 /* S  */ then 1/sum(beta[ii],ii,1,n)/gamma(sum(beta[ii],ii,1,n))*product(gamma(beta[ii]),ii,1,n)            else\n');
    fprintf(of, '  if problem=5 /* U  */ then 2/gamma(sum(beta[ii],ii,1,n))*product(gamma(beta[ii]),ii,1,n),     \n');
    fprintf(of, '  value:radcan(sum(product(ex(x[ii+N*(jj-1)],alpha[jj]),jj,1,n)*x[ii+N*n],ii,1,N)),\n');
    %fprintf(of, '  print("alpha: ",alpha,"  appending e[",kk,"] =",[value-I]),\n');
    fprintf(of, '  eq:append(eq,[value-I])\n');
    fprintf(of,'),\n');
    fprintf(of,'x0:list_matrix_entries(transpose(matrix(\n');
    for i=1:N
        fprintf(of,'  [');
        for j=1:n
            fprintf(of,'%18.15f,',x(i,j));
        end
        if i<N
            fprintf(of,'%18.15f],\n', x(i,n+1));
        else
            fprintf(of,'%18.15f]\n', x(i,n+1));
        end
    end
    fprintf(of,'))),\n');
    ix=find(x==0);                      % indices of zero entries in x
    fprintf(of,'eq:append(eq,[');
    nz=length(ix);
    for i=1:nz
        fprintf(of,'x%d',ix(i));
        if i<nz, fprintf(of,','); end
    end
    fprintf(of,']), /* preserve %d zeros */\n',nz);
    
    fprintf(of, 'load("fsolv.mac"),\n');
    fprintf(of, 'load("plain.mac"),\n');

    fprintf(of, 'print("%s %d_%d_%d"),\n',problems{ip},n,N,deg);
    fprintf(of, 'print(length(x0),"variables, ", length(eq),"equations, %d zeros"),\n',nz);
    fprintf(of, 'solution:fsolv(eq,x,x0,10**(5-fpprec)),\n');
    %fprintf(of, 'print("solution is",solution))$\n');
    fprintf(of, 'plain(x,n,N,deg,"%s"," numerically refined to high precision by %s",err,"%s")\n',...
            problems{ip}, oname, tname);

    fprintf(of, ')$\n');
end
fclose(of);
