function refi(iname, oname, tname)
% refi - read rules from a plain text file.  For each rule, generate
% a Maxima progam that will refine that rule - calculating its
% coefficients and weights to a precision of 36 decimal digits.
% USAGE: refi(iname, oname, tname)
%        refi(iname, deg)
% INPUTS
%   iname = name of input file
%   deg = desired degree
%   oname = optional name of the Maxima program, default "refiXXX.mac",
%           where XXX is the root name of the input file.
%   tname = optional name of file written by the Maxima program, default "refiXXX.txt"
% OUTPUTS
%   refiXXX.mac - the Maxima program,
% where XXX is the root of the input file name

ix=regexp(iname,'\..*$');
if ~isempty(ix)
    root=iname(1:ix-1);
end

desired=0;
if nargin<2
    oname = sprintf('refi%s.mac',root);% Maxima program
elseif ~ischar(oname)
    desired=oname;
    oname = sprintf('refi%s.mac',root);% Maxima program
end

if nargin<3
        tname = sprintf('refi%s.txt',root);% file to collect all the refined rules
end

rules=evaluate(iname);

if length(rules)<1
    fprintf('no valid rules found\n');
    return
end

fprintf('%s creating %s\n',iname,oname);
of=fopen(oname,'w');

for ir=1:length(rules)
    rule=rules(ir);

    x=rule.x;
    n=rule.d;                           % dimensions
    N=rule.n;                           % # points
    if desired>0
        pp=[];
        for o=0:desired
            pp=[pp; nsumk(n,o)];
        end
        deg=desired;
    else
        deg=rule.o;                         % degree
        pp = rule.alpha;                    % powers for monomials correctly integrated
    end

    pstring=rule.problem;
    problems={'G','E2','E1','S','U'};
    for ip=1:length(problems)
        if strcmp(pstring,problems(ip))
            found=1;
            break;
        end
    end
    if ~found
        error('unrecognized problem string "%s"\n', pstring);
        return
    end

    fprintf('formula %d, rule: "%s %d_%d_%d";\n', ir, rule.problem, n, N, deg);

    fprintf(of, '/* %s:%d formula %d */\n', iname, rule.line, ir);
    fprintf(of, 'rule:"%d_%d_%d";\n', n, N, deg);
    fprintf(of, 'n:%d;\n',n);
    fprintf(of, 'N:%d;\n',N);
    fprintf(of, 'deg:%d;\n',deg);

    fprintf(of, 'fpprintprec:32;\n');
    fprintf(of, 'fpprec:fpprintprec*deg+10;\n');

    fprintf(of,'pp:matrix(\n');
    for i=1:size(pp,1)
        fprintf(of, '  [');
        for j=1:n-1
            fprintf(of, '%d, ',pp(i,j));
        end

        if i<size(pp,1), 
            fprintf(of,'%d],\n', pp(i,n));
        else
            fprintf(of, '%d]\n', pp(i,n));
        end
    end
    omax=max(sum(pp,2));
    fprintf(of, ');\n');

    fprintf(of, 'for problem in [%d] do block(\n', ip);
    fprintf(of,'x0:matrix(\n');
    for i=1:N
        fprintf(of,'  [');
        for j=1:n
            fprintf(of,'%18.15f,',x(i,j));
        end
        if i<N
            fprintf(of,'%18.15f],\n', x(i,n+1));
        else
            fprintf(of,'%18.15f]\n', x(i,n+1));
        end
    end
    fprintf(of,'),\n');
    
    fprintf(of, 'load("fsolv2.mac"),\n');
    fprintf(of, 'load("plain.mac"),\n');

    fprintf(of, 'print("%s %d_%d_%d"),\n',problems{ip},n,N,deg);
    fprintf(of, 'print(length(first(x0))-1,"dimensions, ",length(x0),"points, degree=%d"),\n',deg);
    fprintf(of, 'print(%d,"equations, ",%d,"unknowns"),\n',size(pp,1),length(find(x)));
    fprintf(of, '[x,err]:fsolv2(x0,pp,%d,problem,fpprec),\n',omax);
    %fprintf(of, 'print("solution is",solution))$\n');
    %print("have solution, err=",err),
    %fprintf(of, 'print("x=",x),\n');
    fprintf(of, 'print("n=",n,"N=",N,"deg=",deg),\n');
    fprintf(of, 'xli:list_matrix_entries(transpose(x)),\n');
    fprintf(of, 'plain(xli,n,N,deg,"%s"," numerically refined by %s",err,"%s")\n',...
            problems{ip}, oname, tname);

    fprintf(of, ')$\n');
end
fclose(of);



function x = nsumk(n,k)
% nsumk - return an array of n columns, each row of which sums to k
x=zeros(1,n);
x(1,1)=k;
if n>1
    for kr=1:k
        xr=nsumk(n-1,kr);
        [r,c]=size(xr);
        x=[x;[(k-kr)*ones(r,1) xr]];
    end
end
