function plain(x,o,text,oname)
% plain - display an integration rule in plain text
%   x = n * d+1 array.  The first d elements of each row are the
%       coordinates of one integration point.  The last element is the
%       weight for that point.
%   o = the degree or order of the integration rule.  Any
%       polynomial of degree o or less is integrated exactly.
%   text = optional text string to head the written output
%   oname = optional name of the output file.  By default, output
%       is to the display

if nargin<4
    of=1;
else
    of=fopen(oname,'a');
end
if nargin<3; text=''; end

d=size(x,2)-1;                          % dimension
n=size(x,1);                            % # points

radius=sort(round(sqrt(sum(x(:,1:d).^2,2))*1e10)/1e10);
ix = find(diff([-1; radius; 999])); % indices of 1st member of each shell 
count=diff(ix);                         % # members of each shell
shells = length(count);                 % # shells
[foo,jx]=sort(count);
%find(count(jx)>3);
jx=find(count>3);                       % these shells have >3 members
kx=ix(jx);                              % indices of 1st member of each large shell
if length(kx)>3, kx=kx(end-2:end); end  % ...but no more than three of them

y=round(x(:,end)*1e7)/1e7;              % the rounded weights
iy=find(y);                             % indexes of the nonzero weights
y=y(iy);                                % the nonzero weights
yy=round(x(iy,1:d)*1e7)/1e7;            % points with nonzero weights, rounded

central=all(all(matsort(yy)==matsort(-yy)));

baxes=[];
for k=1:d
    [foo,ix]=sort(x(:,k));
    xs=x(ix,:);
    m=sum(xs(:,k)>1e-11);           % count the points on one side
    if m<=n/2
        sign=ones(1,d+1); sign(k)=-1; % reverse sign of kth coordinate
        for i=1:m % look for a mirror image of the ith point
            found=0;
            for i2=n-m+1:n-i+1
                dif=sum((xs(i,:).*sign-xs(i2,:)).^2);
                if dif<1e-11          % found image?
                    temp=xs(n-i+1,:); % swap to end of unmatched points
                    xs(n-i+1,:)=xs(i2,:);
                    xs(i2,:)=temp;
                    found=1;
                    break;
                end
            end
            if ~found; break; end
        end
        if found
            baxes=[baxes k];
        end
    end
end
if length(baxes)==d && central
    symmetry='fully symmetric';
else
    
    if ~isempty(baxes)
    if central 
        symmetry='centrally and ';
    else
        symmetry='';
    end
        symmetry=[symmetry sprintf('bilaterally symmetric (axis %d',baxes(1))];
        for k=2:length(baxes)
            symmetry=[symmetry sprintf(',%d',baxes(k))];
        end
        symmetry=[symmetry ')'];
    elseif central
            symmetry='centrally symmetric';
    else
        symmetry='asymmetric';
    end
end        

if shells>1, rad='radii'; else rad='radius'; end
fprintf(of, '%s %d dimensions, degree %d, %s %d point rule with points at %d distinct %s', ...
        text, d, o, symmetry, n, shells, rad);
if shells<n
    fprintf(of,': %d',count(1));
    for i=2:shells; fprintf(of, '+%d', count(i)); end
    fprintf(of, '=%d\n', n);
end

y=x(:,d+1);
if any(y<0)
    neg=sum(y<0);
    if neg==1, plural=''; else plural='s'; end
    fprintf(of,' (%d negative weight%s, stability factor %f)', ...
            neg, plural, sum(abs(y))/sum(y));
end
fprintf(of,'\n');

%z=sharpen(x);
z=x;

% recalculate with different rules (center point doesn't count as a shell)
%ix = find(diff([0; round(sum(x(:,1:d).^2,2)*1e10); 999])); % indices of 1st member of each shell 
%count=diff(ix);                         % # members of each shell
shells = length(find(count>1));         % # shells


fprintf(of,'#');
for j=1:d; fprintf(of, '                %3s ',sprintf('x%d',j)); end
fprintf(of, '             Weight     Radius');
for i=1:length(kx); fprintf(of,'     cos%d',i); end
fprintf(of,'\n');

for i=1:size(z,1)
    for j=1:size(z,2)
        fprintf(of, ' %19.15f', z(i,j));
    end
    ri=sqrt(sum(z(i,1:d).^2));  % radius to ith point
    fprintf(of, '   %8.5f', ri);
    for ik=1:length(kx)
        k=kx(ik);
        ca = dot(z(i,1:d),z(k,1:d))/sqrt(1.e-10+dot(z(i,1:d),z(i,1:d))*dot(z(k,1:d),z(k,1:d)));
        fprintf(of, ' %8.5f', ca);
    end
    fprintf(of, '\n');
end
fprintf(of, '\n');

if of~=1
    fclose(of);
end


        
function s=matsort(x)
% sort the matrix by rows - i.e. exchange the rows of the matrix so they
% are in order by their first elements, with ties broken by the second
% elements, etc.
[n,m]=size(x);
s=x;
for j=m:-1:1
    [junk,ix]=sort(s(:,j));
    s=s(ix,:);
end
