function nmin=moller(n,d)
% moller - calculate Moller's lower bound on the number of points
% in an integration formula with a spherically symmetric weight and
% region
% USAGE: nmin = moller(n,d)
%        moller
% INPUT: 
%   n = dimensions
%   d = degree
% OUTPUT
%   nmin = lower bound on the number of points in an integration formula
% If no parameters are supplied, a table is printed.

% see: Bo Houd, Panpan Shen, Ran Zhang, Suogang Gao, "On the
% non-existence of tight Gaussian 6-designs on two concentric
% spheres", Discrete Mathematics 313 (2013) 1002-1010.

if nargin>1
    nmin=bound(n,d);
else

    nmin=zeros(20,10);
    for n=1:20
        for d=1:10

            nmin(n,d)=bound(n,d);

        end
    end

    fprintf('  d=');
    for d=1:10
        fprintf(' %7d', d);
    end
    fprintf('\n');
    for n=1:20
        fprintf('n=%2d',n);
        for d=1:10
            fprintf(' %7d', nmin(n,d));
        end
        fprintf('\n');
    end
end


function nmin=bound(n,d)
if ~bitand(d,1) % even degree?
    e=d/2;
    nmin=nchoosek(n+e,e);
else
    s=(d+1)/2;
    nmin=nchoosek(n+s-1,n);
    if bitand(s,1)                      % s is odd?
        for k=1:n-1
            if k+s-2>=k
                nmin=nmin+(1-2^(k-n))*nchoosek(k+s-2,k);
            end
        end
    else                                % s is even
        for k=1:n-1
            if k+s-1>=k
                nmin=nmin+2^(k-n)*nchoosek(k+s-1,k);
            end
        end
    end
end
