function rule=evaluate(name,verbose)
% evaluate - read a rule from a file, determine d, n, test whether it's a
% rule for any of five integrals (and to what order)
% USAGE: rule=evaluate(name,verbose)
% INPUTS
%   name = input filename
%   verbose = negative: don't print anything
%             0: print the integral and order for each rule, and a
%                summary of the number of rules for each integral 
%             1: also print each rule
%             2: also plot the points
%             3: also print the powers, actual moments, and correct moments
%                for that formula
%             4: also print details of determination of d
%             5: for each assumed d, print the calculated radii and the
%                values in the column that should contain the radii
% OUTPUTS
%   rule = the rules, as an array of structs

if nargin<2, verbose=3; end

ifile=fopen(name,'r');
if ifile==-1
    ifile=fopen([name '.txt'],'r');
    if ifile==-1
        error('cannot open %s or %s.txt',name,name);
    end
end

problems={'G','E2','E1','S','U','nothing'};
iline=0;
formula=0;
success=0;
rule=[];
np=length(problems);
found=zeros(1,np);
while 1                                 % find each rule in the file
    recognized=0;
    while 1
        tline=fgetl(ifile);
        iline=iline+1;
        if isempty(tline), continue, end
        if ~ischar(tline)
            if verbose>=0
                
                fprintf('--- summary ---\n');
                for ip=1:np, fprintf('%8s ', problems{ip}); end
                fprintf('\n');
                fprintf('%8d ', found); fprintf('\n');
            end
            return
        end
        vals = textscan(tline, '%f');
        row=vals{1};
        % Matlab's textscan stops scanning if it encounters a bad token.
        % Octave's textscan continues, but records the bad token as NaN.
        % We accommodate either.
        if length(row)>2 && ~any(isnan(row)), break, end
    end

    % we've found the beginning of a rule
    rline=iline;
    i=0;
    x=[];
    while 1
        i=i+1;
        for j=1:length(row)
            x(i,j)=row(j);
        end    
        tline=fgetl(ifile);
        iline=iline+1;
        if isempty(tline), break, end
        if ~ischar(tline), break; end
        vals=textscan(tline,'%f');
        row=vals{1};
        if length(row)<=2, break; end
    end

    formula=formula+1;
    n=size(x,1);                            % # integration points
    if verbose>0, fprintf('# evaluating formula %d with %d points\n', formula, n); end
    d=0;
    for j=3:size(x,2)
        d=j-1;                          % trial degree
        alpha=zeros(1,d);
        beta=(alpha+1)/2;
        wsum = sum(x(:,j));
        m = [2.^(sum(beta,2))/(2*pi)^(d/2).*prod(gamma(beta),2)                    % G volume
            prod(gamma(beta),2)                                                    % E2 volume, Stroud p. 222
            2*factorial(d-1+sum(alpha,2)).*prod(gamma(beta),2)./gamma(sum(beta,2)) % E1 volume, Stroud p. 223
            1./sum(beta,2)./gamma(sum(beta,2)).*prod(gamma(beta),2)                % S volume, Stroud p. 221
            2*pi^(d/2)/d*pi^(d/2)/gamma(d/2)                                       % U, Stroud p 221
            ];                
        if verbose>3 
            fprintf('d=%d? comparing sum of weights %13.6g\n      with: %10.6f %10.6f %10.6f %10.6f %10.6f\n', d, wsum, m);
            fprintf('difference: %10.6f %10.6f %10.6f %10.6f %10.6f\n', wsum-m);
        end 
        if abs(wsum-m(1)) < 1e-6, break; end % G 
        if abs(wsum-m(2)) < 1e-6, break; end % E2
        if abs(wsum-m(3)) < 1e-6, break; end % E1
        if abs(wsum-m(4)) < 1e-6, break; end % S 
        if abs(wsum-m(5)) < 1e-6, break; end % U
        if verbose>4, fprintf('checking whether rss of first %d cols equal col %d:\n',j-2,j); end
        rss=sqrt(sum(x(:,1:j-2).^2,2));
        if verbose>5, disp([rss, x(:,j)]); end
        d=0;
        if mean(abs(rss-x(:,j)))<.001
            d=j-2;
            break
        end
    end
    if d
        if verbose>3, fprintf('found d=%d\n', d); end
        x=x(:,1:d+1);                   % discard the extra columns
        
    else
        d=size(x,2)-1;                  % there are no extra columns
        if verbose>3, fprintf('trying d=%d (all columns)\n', d); end
    end


    for ip=1:np-1                       % check each of the integrals
        problem=problems{ip};
        switch problem
          case 'G'
            V=1;
          case 'E2'
            V=pi^(d/2); % Stroud, p, 222
          case 'E1'
            V=2*factorial(d-1)*pi^(d/2)/gamma(d/2);     % Stroud p 223
          case 'S'
            V=2/d*pi^(d/2)/gamma(d/2); % Stroud p 220
          case 'U'                      % the surface of S_n
            V=2*pi^(d/2)/d*pi^(d/2)/gamma(d/2); % Stroud p 221
        end
        ppp=[];
        w=x(:,d+1);
        stability=sum(abs(w))/sum(w);   % account for cancellation
        for o=0:19                      % try all the orders
            pp=nsumk(d,o);              % powers for all constraints of this order
            ix=1:(d+1)*n;
            e=mce(x',d,n,ix,pp,problem,verbose);
            lx=abs(e)<V*stability*1e-7;
            ppp=[ppp; pp(lx,:)];
            pass=sum(lx);
            if pass < length(e), 
                break; 
            end
        end
        
        if o>2
            o=o-1;
            if verbose >= 0
                fprintf('# %s:%d formula %-4d: problem %2s, %d_%d_%d  (d=%d n=%2d o=%d plus %d/%d of order %d)', ...
                        name, rline, formula, problem, d, n, o, d, n, o, pass, length(e), o+1);
                
                w=x(:,d+1);
                zer=sum(abs(w)<1e-10);
                if zer
                    if zer==1, plural=''; else plural='s'; end
                    fprintf(' (%d non-zero weight%s)', size(w,1)-zer, plural);
                end
                neg=sum(w<0);
                if neg
                    quality='N';
                else
                    if all(abs(w-w(1))<V*1e-7)
                        quality='E';
                    else
                        quality='P';
                    end
                end

                switch problem
                  case 'S'
                    r=sqrt(sum(x(:,1:d).^2,2));
                    if all(r<1-1e-7)
                        quality=[quality 'I'];
                    else
                        if any(r>1+1e-7)
                            quality=[quality 'O'];
                        else
                            if any(r>1-1e-7)
                                quality=[quality 'B'];
                            end
                        end
                    end
                end
                fprintf(' quality %s', quality);
                
                if neg
                    if neg==1, plural=''; else plural='s'; end
                    fprintf(' (%d negative weight%s, stability factor %f)', ...
                            neg, plural, sum(abs(w))/sum(w));
                end
                fprintf('\n');
            end
            if verbose>0, plain(x,o,'#'); end
            recognized=1;
            found(ip)=found(ip)+1;

            success=success+1;
            rule(success).x=x;
            rule(success).d=d;
            rule(success).n=n;
            rule(success).o=o;
            rule(success).problem=problem;
            rule(success).alpha=ppp;
            rule(success).filename=name;
            rule(success).line=rline;
            if verbose>1 
                if d>=3
                    plot3(x(:,1),x(:,2),x(:,3),'r*'); % for higher dimensions, we plot a projection
                    lim=max([x;-x])*1.05;
                    axis([-lim(1) lim(1) -lim(2) lim(2) -lim(3) lim(3)]);
                    axis equal;
                    xlabel('x');
                    ylabel('y');
                    zlabel('z');
                    grid on;
                    %if strcmp(problem,'S')
                    %    a=0:pi/30:2*pi;
                    %    hold on;
                    %    plot(sin(a),cos(a),'k--');
                    %    hold off;
                    %end
                else
                    plot(x(:,1),x(:,2),'r*'); % for 2 dimensions
                    lim=max([x;-x])*1.05;
                    axis([-lim(1) lim(1) -lim(2) lim(2)]);
                    axis equal;
                    if strcmp(problem,'S')
                        a=0:pi/30:2*pi;
                        hold on;
                        plot(sin(a),cos(a),'k--');
                        hold off;
                    end
                %else
                %    plot(sqrt(sum(x(:,1:d).^2,2)), x(:,d+1), 'b*');
                %    pause;
                input('hit ENTER to look for next rule');
                end
            end
        end
    end
    if ~recognized
        fprintf('# %s:%d formula %-4d:   nothing   %d_%d_*\n', ...
                            name, rline, formula, d, n); 
        if verbose>0, plain(x,0,sprintf('unrecognized formula %d:',formula)); end
        found(np)=found(np)+1;
            if verbose>1 
                if d>=3
                    plot3(x(:,1),x(:,2),x(:,3),'r*'); % for higher dimensions, we plot a projection
                    lim=max([x;-x])*1.05;
                    axis([-lim(1) lim(1) -lim(2) lim(2) -lim(3) lim(3)]);
                    axis equal;
                    xlabel('x');
                    ylabel('y');
                    zlabel('z');
                    grid on;
                    %if strcmp(problem,'S')
                    %    a=0:pi/30:2*pi;
                    %    hold on;
                    %    plot(sin(a),cos(a),'k--');
                    %    hold off;
                    %end
                else
                    plot(x(:,1),x(:,2),'r*'); % for 2 dimensions
                    lim=max([x;-x])*1.05;
                    axis([-lim(1) lim(1) -lim(2) lim(2)]);
                    axis equal;
                    if strcmp(problem,'S')
                        a=0:pi/30:2*pi;
                        hold on;
                        plot(sin(a),cos(a),'k--');
                        hold off;
                    end
                %else
                %    plot(sqrt(sum(x(:,1:d).^2,2)), x(:,d+1), 'b*');
                %    pause;
                end
                input('hit ENTER to look for the next rule');
            end
    end
end


function x = nsumk(n,k)
% nsumk - return an array of n columns, each row of which sums to k
x=zeros(1,n);
x(1,1)=k;
if n>1
    for kr=1:k
        xr=nsumk(n-1,kr);
        [r,c]=size(xr);
        x=[x;[(k-kr)*ones(r,1) xr]];
    end
end

function e = mce(x,d,n,ix,alpha,problem,verbose)

% moment(k+1) is the value of the integral when the integrated function
% is x^k.  The first element tells us that the weights sum to 1.
% moment = [1 0 1 0 3 0 15 0 105 0 945 0 ...
omax=max(sum(alpha,2));
moment = zeros(1,omax+1);
moment(1) = 1;
moment(3:2:omax+2) = cumprod(1:2:omax);

% The first d elements of each column are the coordinates of a
% sigma point.  The last element is the weight for that sigma point.
xx=zeros(d+1,n);
xx(ix)=x;
n=size(xx,2);
w=xx(end,:)';
K=size(alpha,1);
e=zeros(K,1);
P=zeros(d+1,n);
J=zeros(K,size(ix,1));
%w=2*xx(end,:)';
%if center(1), w(1)=w(1)/2; end
e=zeros(K,1);

% prepare an array with all the needed powers of all the point coordinates
%  xxx(i,j,k+2)=x_{i,j}^k where i=dimension, j=point
xxx=zeros(d+1,n,2+omax);
xxx(:,:,2)=1;
xxx(:,:,3)=xx;
for ip=2:omax+1
    xxx(:,:,ip+2)=xxx(:,:,ip+1).*xx;
end


v2=zeros(d,n);
ne=size(alpha,1);
v=zeros(ne,n);
for ie=1:ne
    for i=1:d
        v2(i,:)=xxx(i,:,alpha(ie,i)+2);
    end
    v(ie,:)=prod(v2);
end

m=zeros(K,1);
% the faster if more obscure expressions
switch problem
  case 'G'
    m=prod(moment(alpha+1),2);
  case 'E2'
    beta=(alpha+1)/2;
    m=(2*pi)^(d/2)./2.^sum(beta,2).*prod(moment(alpha+1),2);
  case 'E1'
    beta=(alpha+1)/2;
    m=pi^((d-1)/2)*2.^(sum(alpha,2)/2+d).*gamma(sum(beta,2)+.5).*prod(moment(alpha+1),2);
  case 'S'
    beta=(alpha+1)/2;
    m=(2*pi)^(d/2)./2.^(sum(beta,2))./gamma(sum(beta,2)+1).*prod(moment(alpha+1),2);
  case 'U'
    beta=(alpha+1)/2;
    m=2*(2*pi)^(d/2)./2.^(sum(beta,2)).*sum(beta,2)./gamma(sum(beta,2)+1).*prod(moment(alpha+1),2);
end

e=v*w-m; % evaluate all the constraints at once
if verbose>2
    fprintf('%s moment constraints of order %d\n',problem,max(max(alpha)));
    mm=v*w;
    for i=1:length(m)
        fprintf('%d ', alpha(i,:));
        fprintf('%9.6f %9.6f\n', mm(i), m(i));
    end%    disp([alpha v*w m]);
end
%if nargout>1  % caller wants the Jacobian?
%
%    J=zeros(K,size(ix,1)); % J(k,j) is the derivative of the kth constraint
%                           % with respect to the jth free parameter
%    P=zeros(d+1,n);
%    
%    v2=zeros(d+1,n);
%    for ie=1:ne                         % calculate row ie of the Jacobian 
%                                        % (derivatives of ieth constraint)
%        p=alpha(ie,:);
%        for jd=1:d                      % calculate the jth row of P
%                                        % (the derivatives with
%                                        % respect to the jdth coordinate)
%            v2(d+1,:)=p(jd)*w;
%            
%            p(jd)=p(jd)-1;
%            for i=1:d
%                v2(i,:)=xxx(i,:,p(i)+2);
%            end
%            p(jd)=p(jd)+1;
%            
%            P(jd,:)=prod(v2);
%        end
%        %fprintf('ie=%d  p=[',ie); fprintf('%d ',p); fprintf(']\n')
%        %fprintf('P=\n'); disp(P);
%        P(d+1,:)=2*v(ie,:);             % the last row of P has the derivatives
%                                        % with respect to the weights
%        %if center(1)
%        %    P(d+1,1)=P(d+1,1)/2;
%        %end
%        J(ie,:)=P(ix)';                 % select the derivatives
%                                        % for one row of the Jacobian
%    end
%end


