function [x,fval,flag,output]=UDL(f,x,options)
% UDL - return a root of function f near x using the
%   Under-Determined Dogleg method.
% USAGE: [x,fval,flag,output]=UDL(f,x,options)
% INPUTS
%   f function with calling sequence
%     [F, B] = f(x);
%     F = function value
%     B = Jacobian
%   x            initial guess of solution
%   options      struct with optimization options (all optional)
%     TolFun       solution tolerance
%     MaxIter      maximum number of iterations
% OUTPUTS
%   x = final estimate of root
%   fval = residual vector at x
%   flag: 1 = converged to a solution (residual absolute error < TolFun)
%         0 = iterations exceeded MaxIter
%         3 = convergence too slow
%   output: struct with these elements:
%     funcCount = count of calls to f
%     iterations = count of iterations of the algorithm
%     resids = array of norm(F) on each iteration
%     funcalls = array of cumulative # function calls at end of each iteration

% Author: Joseph Simonis <joseph.simonis@gmail.com>
% Latest update: 03-01-2006
% 05-03-2017 Jim Van Zandt: f can return both the residuals and the Jacobian.
% 07-01-2017 Jim Van Zandt: Calling sequence matches fsolve().  
% 07-17-2017 Jim Van Zandt: Fail if norm of residual decreases by less
% than 5% over 5 iterations.
% 09-22-2015 Jim Van Zandt: Include Dogleg() locally.  Add self-test code for Octave.
% 10-04-2017 Jim Van Zandt: Fail if norm of residual decreases by less
% than 7% over 7 iterations.
if nargin<3, options=[]; end
maxits      = fetchOption(options, 'MaxIter',200);
tol         = fetchOption(options, 'TolFun',10*eps);

t=10^-4;
thetamin = 0.1;
thetamax = 0.5;
u=.75;
v=0.1;
inneritsmax=20;
%
evals=0;

% Algorithm
F=feval(f,x);
evals=evals+1;
residual = norm(F);
%fprintf('\nIt.No. ||F(u)|| GMRES Its. Lin Mod Norm Delta \n');
%fprintf(' %d %e %c %e %e\n', 0,residual,'*',0,0);
its=1;
innerits=0;
resids(its)=residual;
funcalls(its)=evals;
flag=0;                                 % assume we'll run out of iterations
while(residual > tol && its<maxits)
    [F,J]=feval(f,x);
    evals=evals+1;
    % Calculate the Moore-Penrose step.
    snewt=-pinv(full(J))*F;
    snewtnorm=norm(snewt);
    if (its==1)
        delta=snewtnorm;
    end
    % Calculate the dogleg_step.
    dogleg_step = Dogleg(F,J,snewt,snewtnorm,delta);
    Fpls = feval(f,x+dogleg_step);
    evals=evals+1;
    Fplsn = norm(Fpls);
    Js = J*dogleg_step;
    lin_res = norm(F+Js);
    ared = residual-Fplsn;
    pred = residual-lin_res;
    % Inner Dogleg loop.
    while (ared<t*pred && innerits < inneritsmax);
        if (snewtnorm < delta)
            delta = snewtnorm;
        end
        d = Fplsn^2-residual^2-2*F'*Js;
        if (d <= 0)
            theta = thetamax;
        else
            theta = -(F'*Js)./d;
            if (theta > thetamax)
                theta = thetamax;
            end
            if (theta < thetamin)
                theta = thetamin;
            end
        end
        delta = theta*delta; % Update Delta
                             % Recalculate dogleg_step.
        dogleg_step = Dogleg(F,J,snewt,snewtnorm,delta);
        Fpls = feval(f,x+dogleg_step);
        evals=evals+1;
        Fplsn = norm(Fpls);
        Js = J*dogleg_step;
        lin_res = norm(F+Js);
        ared = residual-Fplsn;
        pred = residual-lin_res;
        innerits=innerits+1;
    end
    innerits = 0;
    x=x+dogleg_step;
    F=Fpls;
    residual=Fplsn;
    %fprintf(' %d %e %c %e %e\n', its,residual,'*',lin_res,delta);
    its = its+1;
    resids(its)=residual;
    funcalls(its)=evals;
    if its>3 && resids(its-2)-residual<tol, flag=3; break; end % improvement too small
    if its>8 && resids(its-8)/residual < 1.07, flag=3; break; end  % improvement too slow

    % Update delta
    if (ared > u.*pred && snewtnorm > delta)
        delta = 2.*delta;
    elseif (ared < v.*pred)
        delta = .5.*delta;
    end
end

fval=F;
if residual<=tol; flag=1; end           % success!
output.funcCount=evals;
output.iterations=its;
output.resids=resids;
output.funcalls=funcalls;

% global diagnostics
% if ~isempty(diagnostics) && diagnostics
%     of=fopen([mfilename '.stats'],'a');
%     fprintf(of,'\n\n');
%     fprintf(of,'# %s %d equations  %d unknowns\n',mfilename,length(F),length(x));
%     for i=1:length(resids)
%         fprintf(of,'%d %g\n',funcalls(i),resids(i));
%     end
%     fclose(of);
%     %output.fnorms=resids;
% end


function opt=fetchOption(options,field,default)
if isa(options,'struct') && isfield(options,field)
    opt=getfield(options,field);
    if isempty(opt)
        opt=default;
    end
else
    opt=default;
end



function [dogleg_step]=Dogleg(F,J,snewt,snewtnorm,delta)
% Computes the dogleg_step
% Inputs:
%     F = F(xcurrent)
%     J = J(xcurrent)
%     snewt = Moore-Penrose Step
%     snewtnorm = norm of Step
%     delta = current trust region radius

if (snewtnorm <= delta)
    dogleg_step = snewt;
else
    JTF=J'*F;
    JTFnorm=norm(JTF);
    JJTFnorm=norm(J*JTF);
    sdescent = -(JTFnorm./JJTFnorm)^2.*JTF;
    sdescentnorm = norm(sdescent);
    if (sdescentnorm >= delta)
        dogleg_step = (delta./sdescentnorm).*sdescent;
    else
        sdiff = snewt-sdescent;
        a = norm(sdiff).^2;
        b = sdescent'*sdiff;
        c = sdescentnorm.^2-delta.^2;
        tao = -c./(b+sqrt(b.^2-a*c));
        dogleg_step = sdescent + tao.*sdiff;
    end
end


%  self-test code, exercised under Octave with:
%    test UDL
%
%!function [F,J]=fct(x)
%!  F=sin(x);
%!  J=cos(x);
%!endfunction
%!assert(UDL('fct',3),pi,eps)
%   Beal function, see: Jorge J. Moré, Burton S. Garbow, and Kenneth
%   E. Hillstrom. Testing unconstrained optimization software. ACM
%   Trans. Math. Softw., 7(1):17--41, March 1981.
%!function [F,J]=beale(x,flag)
%!  y=[1.5; 2.25; 2.625];
%!  F=[ y(1) - x(1)*(1 - x(2)^1)
%!      y(2) - x(1)*(1 - x(2)^2)
%!      y(3) - x(1)*(1 - x(2)^3) ];
%!  J=[ x(2)-1      x(1)	 
%!      0           3*x(2)**2-1 
%!      x(2)**3-1   3*x(1)*x(2)**2 ];
%!endfunction
%!assert(UDL('beale',[1;1]),[3;0.5],10*eps)
